/*
    Windows:
        cl /W4 /EHsc /Iapi api\api_client_test.cpp api\client.cpp
    Mac OS:
        clang++ -o api_client_test -std=c++11 -I. api/api_client_test.cpp api/client.cpp
 */

#include "api/client.h"
#include "api/container.h"
#include "api/shader.h"

#include <iostream>

int main(int argc, char** argv)
{
    if ( argc != 5 ) {
        std::cout<<"Usage: "<<argv[0]<<" <TFD library file> <cache base path> <frame number> <channel name>"<<std::endl;
        return 1;
    }

    if ( !tfd::api::load_library(argv[1]) ) {
        std::cout<<"Failed to load TFD shared library from "<<argv[1]<<std::endl;
        return 1;
    }

    if ( tfd::api::CLIENT_VERSION > tfd::api::get_server_version() ) {
        std::cout<<"Version mismatch: client = "<<tfd::api::CLIENT_VERSION<<" server = "<<tfd::api::get_server_version()<<std::endl;
        return 1;
    }

    int frame_number = atoi(argv[3]);
    tfd::ChannelContainer
        container(argv[2], frame_number);
    if ( !container.valid() ) {
        std::cout<<"Failed to load channel container"<<std::endl;
        return 1;
    }

    auto grid_offset = container.grid_offset();
    std::cout<<"Container grid offset ["<<grid_offset.x<<","<<grid_offset.y<<","<<grid_offset.z<<"]"<<std::endl;

    auto voxel_size = container.voxel_size();
    std::cout<<"Container voxel size "<<voxel_size<<std::endl;

    auto x = 0.5f * (container.width()-1);
    auto y = 0.5f * (container.height()-1);
    auto z = 0.5f * (container.depth()-1);

    auto channel = container.get_channel(argv[4]);
    if ( !channel ) {
        std::cout<<"Channel container has no channel named "<<argv[4]<<std::endl;
        return 1;
    }

    auto scalar = channel->sample_linear(x, y, z);
    std::cout<<"Scalar channel value "<<scalar<<std::endl;

    auto mapping_str = tfd::String("(0.005019,0,0.303265,1,0),(0.067979,0.508929,0.5,1,1),(0.1,1,0.236183,0,1),(0.361604,0,0.5,1,1)");
    auto gradient_str = tfd::String(
"0,1,[0,0,0],[0,0,0],([0,0,0],[0.000387228,1.01746e-005,0],[0.00043786,1.19638e-005,0],[0.000494469,1.40289e-005,0],[0.000557684,1.64074e-005,0],[0.000628191,1.9141e-005,0],[0.000706737,2.22761e-005,0],[0.000794134,2.58648e-005,0],\
[0.00215316,9.04959e-005,0],[0.00239241,0.000103067,0],[0.00265551,0.000117192,0],[0.00294453,0.000133041,0],[0.00326174,0.000150801,0],[0.00360955,0.000170672,0],[0.00399055,0.000192877,0],[0.00440752,0.000217658,0],[0.00486345,0.000245277,0],\
[0.0635662,0.00518465,0],[0.0684892,0.00565611,0],[0.0737476,0.00616557,0],[0.079361,0.00671568,0],[0.0853499,0.00730926,0],[0.0917358,0.00794928,0],[0.0985411,0.00863891,0],[0.105789,0.00938145,0],[0.113505,0.0101804,0],[0.121714,0.0110396,0],\
[0.971383,0.120447,0],[1.02471,0.128029,0],[1.08056,0.136027,0],[1.13901,0.144458,0],[1.20018,0.153344,0],[1.26416,0.162705,0],[1.33107,0.172561,0],[1.40101,0.182936,0],[1.4741,0.193853,0],[1.55044,0.205334,0],[1.63017,0.217404,0],[1.71339,0.230088,0],\
[8.70765,1.45403,0],[9.05179,1.51911,0],[9.40726,1.58665,0],[9.77436,1.65675,0],[10.1534,1.72946,0],[10.5446,1.80488,0],[10.9484,1.88307,0],[11.365,1.96413,0],[11.7947,2.04814,0],[12.2379,2.13518,0],[12.695,2.22533,0],[13.1661,2.3187,0],\
[38.1914,7.80632,0.171365],[39.3746,8.08742,0.190491],[40.5879,8.37703,0.210718],[41.8316,8.67536,0.23209],[43.1066,8.98262,0.254655],[44.4132,9.29901,0.27846],[45.7522,9.62476,0.303555],[47.124,9.96008,0.32999],[48.5294,10.3052,0.357818],\
[104.168,24.9725,1.96669],[106.84,25.7162,2.06529],[109.566,26.4781,2.16769],[112.349,27.2587,2.27399],[115.188,28.0583,2.38431],[118.084,28.8773,2.49877],[121.038,29.716,2.61748],[124.052,30.5749,2.74057],[127.125,31.4541,2.86816],\
[202.51,53.9465,6.58337],[207.045,55.3489,6.83945],[211.661,56.781,7.10352],[216.357,58.2434,7.37576])");

    tfd::ShaderParameters
        shader_params;
    *shader_params.fire_color_channel_name = tfd::String(argv[4]);
    *shader_params.fire_color_mapping = mapping_str;
    *shader_params.fire_color_gradient = gradient_str;
    shader_params.fire_opacity = 1.f;

/*
    // Exporting TFD shader parameters from C4D scene using lib_tfd:
    TFDFluidContainer*
        container = ...;
    tfd::ShaderParameters
        shader_params;
    container->get_shader_parameters(shader_params);
*/

    tfd::Shader
        shader(container, shader_params);
    float rgbe[4];
    shader.shade_fire(x, y, z, rgbe);
    std::cout<<"Shaded RGB ["<<rgbe[0]<<","<<rgbe[1]<<","<<rgbe[2]<<"] extinction "<<rgbe[3]<<std::endl;
    return 0;
}
